const { genSaltSync, hashSync, compareSync } = require("bcrypt");
const { sign } = require("jsonwebtoken");
const crypto = require("crypto");
const validator = require("../middleware/validator");

const {
  create,
  user,
  getUserByUserEmail,
  contact,
  checkIfEmailExists,
  addNewSubscriber,
  sandWelcomeEmail,
  sendBrocherEmail,
} = require("../model/service");

// ✅ Secure brochure link generator
const generateSecureLink = (email) => {
  const token = crypto.randomBytes(32).toString("hex");
  const baseUrl = "http://localhost:3200/download/saleskit.pdf"; // Replace with actual download route
  return `${baseUrl}?token=${token}&email=${encodeURIComponent(email)}`;
};

module.exports = {
  // ✅ User signup
  createUser: (req, res) => {
    const body = req.body;
    const salt = genSaltSync(10);
    body.password = hashSync(body.password, salt);
    user(body, (err, results) => {
      if (err) {
        console.log(err);
        return res.status(500).json({
          status: "error",
          message: "Database connection error !!",
        });
      }
      return res.status(200).json({
        status: "success",
        data: results,
      });
    });
  },

  // ✅ User login
  login: (req, res) => {
    const body = req.body;
    getUserByUserEmail(body.email, (err, results) => {
      if (err) {
        console.log(err);
      }
      if (!results) {
        return res.json({
          status: false,
          data: "invalid email or password !",
        });
      }
      const result = compareSync(body.password, results.password);
      if (result) {
        results.password = undefined;
        const jsontoken = sign({ result: results }, "qwe1234", {
          expiresIn: "24h",
        });
        return res.json({
          status: true,
          message: "Login Successfully !",
          token: jsontoken,
        });
      } else {
        return res.json({
          status: false,
          data: "invalid email or password !",
        });
      }
    });
  },

  // ✅ Enquiry form handler
createEnquiry: [
  validator.validateEnquiry,
  async (req, res) => {
    const body = req.body;

    create({ ...body }, async (err, results) => {
      if (err) {
        console.error("Error inserting/updating enquiry:", err);

        return res.status(500).json({
          status: "error",
          message: "Something went wrong while processing the enquiry.",
        });
      }

      return res.status(200).json({
        status: "success",
        message: results.affectedRows === 1
          ? "Enquiry submitted successfully."
          : "Enquiry updated successfully.",
        data: results,
      });
    });
  },
],


contact: [
  validator.validateContact, // Assuming you have a validator middleware
  async (req, res) => {
    const body = req.body;

    contact({ ...body }, async (err, results) => {
      if (err) {
        console.error("Error inserting/updating contact:", err);

        return res.status(500).json({
          status: "error",
          message: "Something went wrong while processing the contact form.",
        });
      }

      const isUpdate = results.affectedRows === 2;

      return res.status(200).json({
        status: "success",
        message: isUpdate
          ? "Contact details updated successfully."
          : "Contact submitted successfully.",
        data: results,
      });
    });
  },
],


  // ✅ Brochure subscribe handler with secure link email
subscribeBrochure: (req, res) => {
  const subscribe_email = req.body.email?.toLowerCase();
  const source = req.body.source?.toLowerCase(); // Accepts "frealestate" or "ftvbar"

  if (!subscribe_email || !source) {
    return res.status(400).json({
      status: "error",
      message: "Email and source are required.",
    });
  }

  // Choose file based on source
  let filePath = "";
  if (source === "frealestate") {
    filePath = "http://localhost:3200/download/realestate-saleskit.pdf";
  } else if (source === "ftvbar") {
    filePath = "http://localhost:3200/download/ftvbar-saleskit.pdf";
  } else {
    return res.status(400).json({
      status: "error",
      message: "Invalid source provided.",
    });
  }

  checkIfEmailExists(subscribe_email, (err, results) => {
    if (err) {
      return res.status(500).json({
        status: "error",
        message: "Database Connection Error !!",
      });
    }

    const subscriberData = {
      email: subscribe_email,
      source: source,
    };

    addNewSubscriber(subscriberData, (err) => {
      if (err) {
        return res.status(400).json({
          status: "error",
          message: "Error while saving subscriber to the database",
        });
      }

      sendBrocherEmail(subscribe_email, filePath, (err) => {
        if (err) {
          return res.status(500).json({
            status: "error",
            message: "Error while sending brochure email",
          });
        }

        return res.status(200).json({
          status: "success",
          message: `Secure brochure link sent via email for '${source}'`,
        });
      });
    });
  });
}

};
